<?php
// ==============================================
// CONFIGURATION INITIALE
// ==============================================

// Désactive l'affichage des erreurs et active le logging
ini_set('display_errors', '0');
ini_set('log_errors', '1');
ini_set('error_log', __DIR__.'/../../logs/api_errors.log');

// Démarre le buffer de sortie
ob_start();

// Vérifie que la requête est bien AJAX
if (!isset($_SERVER['HTTP_X_REQUESTED_WITH']) || strtolower($_SERVER['HTTP_X_REQUESTED_WITH']) !== 'xmlhttprequest') {
    http_response_code(403);
    die('{"success":false,"error":"Accès direct interdit"}');
}

// Force le header JSON
header('Content-Type: application/json; charset=UTF-8');

// ==============================================
// TRAITEMENT PRINCIPAL
// ==============================================

$response = ['success' => false];
$pdo = null;

try {
    // 1. Vérification de la méthode HTTP
    if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
        throw new Exception("Méthode non autorisée", 405);
    }

    // 2. Récupération des données
    $jsonInput = file_get_contents('php://input');
    $data = json_decode($jsonInput, true);

    if (json_last_error() !== JSON_ERROR_NONE) {
        throw new Exception("Données JSON invalides", 400);
    }

    // 3. Validation des champs obligatoires
    $requiredFields = ['employee_id', 'period', 'gross_salary'];
    foreach ($requiredFields as $field) {
        if (empty($data[$field])) {
            throw new Exception("Le champ $field est obligatoire", 400);
        }
    }

    $employeeId = (int)$data['employee_id'];
    if ($employeeId <= 0) {
        throw new Exception("ID employé invalide", 400);
    }

    // 4. Connexion à la base de données
    require_once __DIR__.'/../../models/Database.php';
    require_once __DIR__.'/../../utils/Helpers.php';
    $pdo = Models\Database::getInstance()->getConnection();
    $pdo->beginTransaction();

    // 5. Calcul des valeurs
    $period = htmlspecialchars($data['period']);
    $gross = (float)$data['gross_salary'];
    //$imposable = (float)($data['imposable'] ?? 0);

    //$tauxImpot = (float)($data['taux_impot'] ?? 0);
    $tauxImpot = (float)str_replace(',', '.', $data['taux_impot'] ?? 0);

    $otherPlus = 0;
    $otherMinus = 0;

    if (!empty($data['plus'])) {
        foreach ($data['plus'] as $line) {
            $otherPlus += $line['amount'];
        }
    }

    if (!empty($data['minus'])) {
        foreach ($data['minus'] as $line) {
            $otherMinus += $line['amount'];  // déjà négatif normalement
        }
    }
    $cotisations = calculCotisationsSociales($gross);
    $socialCharges = $cotisations['total'];  // somme réelle des cotisations sociales

    $totalAbattements = 0;
    if (!empty($data['moderation_lines'])) {
        foreach ($data['moderation_lines'] as $line) {
            $totalAbattements += $line['amount'];
        }
    }

    // Salaire imposable = brut – cotisations – abattements
    $imposable = $gross - $socialCharges - $totalAbattements;

    // Impôt
    $impot = calculImpots($imposable, $tauxImpot);

    // Net avant avant impôts
    $netSalary = $gross - $socialCharges;
    // Net avant autres ajustements
    $net = $gross - $socialCharges - $impot;

    // Puis ajoute les plus et moins
    $netFinal = $net + $otherPlus + $otherMinus;

    $year = (int)substr($period, -4);

    // 6. Insertion dans la table principale
    $stmt = $pdo->prepare("
                INSERT INTO payslips 
                (employee_id, period, gross_salary, social_charges, abattements, taux_impot, imposable, impots, other_plus, other_minus, net_salary, net_apres_impot, net_final, year)
                VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
            ");


    $stmt->execute([
        $employeeId, $period, $gross,
        $socialCharges, $totalAbattements,
        $tauxImpot, $imposable, $impot,
        $otherPlus, $otherMinus, $netSalary, $net, $netFinal, $year
    ]);
    $payslipId = $pdo->lastInsertId();

// 7. Insertion des cotisations sociales

    if (!empty($cotisations['cotisations'])) {
        $stmtSocial = $pdo->prepare("
            INSERT INTO payslip_social 
            (payslip_id, employee_id, period, keyname, label, rate, gross_base, amount) 
            VALUES (?, ?, ?, ?, ?, ?, ?, ?)
        ");
        $file = fopen(__DIR__ . '/../../logs/store_payslip.txt', 'w');
        foreach ($cotisations['cotisations'] as $social) {
            $keyname = strtoupper($social['keyname']);
            fwrite($file, 'Salaire brut de base pour '. $social['label'] . ' égal à ' . $social['base'] . '\n');
            $stmtSocial->execute([
                $payslipId,
                $employeeId,
                $period,
                $keyname,
                $social['label'],
                $social['rate'],
                $social['base'],    // ✅ base réelle calculée
                $social['amount']
            ]);
        }
        fclose($file);
    }

    // 8. Insertion des abattements
    if (!empty($data['moderation_lines'])) {
        $stmtAbatt = $pdo->prepare("INSERT INTO payslip_moderation (payslip_id, employee_id, keyname, label, amount, period)
                                    VALUES (?, ?, ?, ?, ?, ?)");
        foreach ($data['moderation_lines'] as $line) {
            $stmtAbatt->execute([
                $payslipId,
                $employeeId,
                $line['key'] ?? '',
                $line['label'] ?? '',
                $line['amount'] ?? 0,
                $period
            ]);
        }
    }

    // 9. Insertion des taxes
    if (!empty($data['taux_impot']) && !empty($data['montant_impot'])) {
        $stmtTax = $pdo->prepare("INSERT INTO payslip_taxes 
                                 (payslip_id, employee_id, base_gross, tax_rate, tax_amount, period)
                                 VALUES (?, ?, ?, ?, ?, ?)");
        $stmtTax->execute([
            $payslipId, $employeeId, $data['salaire_imposable'], $data['taux_impot'], $data['montant_impot'], $period
        ]);
    }

    // 10. Insertion des valeurs supplémentaires et déductions
    if (!empty($data['plus'])) {
        $stmtPlus = $pdo->prepare("INSERT INTO payslip_plus_minus (payslip_id, employee_id, keyname, label, amount)
                                   VALUES (?, ?, ?, ?, ?)");
        foreach ($data['plus'] as $line) {
            $stmtPlus->execute([
                $payslipId,
                $employeeId,
                substr($line['label'], 0, 4),
                $line['label'],
                $line['amount']
            ]);
        }
    }

    if (!empty($data['minus'])) {
        $stmtMinus = $pdo->prepare("INSERT INTO payslip_plus_minus (payslip_id, employee_id, keyname, label, amount)
                                    VALUES (?, ?, ?, ?, ?)");
        foreach ($data['minus'] as $line) {
            $stmtMinus->execute([
                $payslipId,
                $employeeId,
                substr($line['label'], 0, 4),
                $line['label'],
                $line['amount']
            ]);
        }
    }

    // 11. Commit et préparation de la réponse
    $pdo->commit();
    
    $response = [
        'success' => true,
        'payslip_id' => $payslipId,
        'insertions' => [
            'payslips' => true,
            'payslip_social' => true,
            'payslip_taxes' => isset($stmtTax),
            'payslip_moderation' => !empty($data['moderation_lines'])
        ]
    ];

} catch (PDOException $e) {
    // Gestion spécifique des erreurs SQL
    if (isset($pdo) && $pdo->inTransaction()) {
        $pdo->rollBack();
    }
    $response['error'] = "Erreur base de données";
    $response['debug'] = $e->getMessage();
    http_response_code(500);

} catch (Exception $e) {
    // Gestion des autres exceptions
    $response['error'] = $e->getMessage();
    $response['code'] = $e->getCode() ?: 500;
    http_response_code($e->getCode() ?: 500);

} finally {
    // Nettoyage final et envoi de la réponse
    ob_end_clean();
    echo json_encode($response);
    exit;
}